use anyhow::{bail, Result};
use clap::ArgMatches;
use config::CompressionType;
use config::Config;
use config::ContainerType;
use ignore::WalkBuilder;
use indicatif::{ProgressBar, ProgressIterator, ProgressStyle};
use log::LevelFilter;
use log::{debug, error, info, warn};
use path_slash::PathExt;
use relative_path::RelativePath;
use simplelog::ColorChoice;
use simplelog::ConfigBuilder;
use simplelog::TermLogger;
use simplelog::TerminalMode;
use std::env::{current_dir, set_current_dir};

mod config;
mod container;

fn main() -> Result<()> {
    #[cfg(windows)]
    colored::control::set_virtual_terminal(true).unwrap();

    // Args
    let matches = config::create_app().get_matches();

    // Init logs
    let level_filter = match matches.get_count("verbosity") {
        0 => LevelFilter::Info,
        1 => LevelFilter::Debug,
        _ => LevelFilter::Trace,
    };
    TermLogger::init(
        level_filter,
        ConfigBuilder::new()
            .set_time_offset_to_local()
            .unwrap()
            .build(),
        TerminalMode::Stderr,
        ColorChoice::Auto,
    )?;

    if let Err(e) = run(&matches) {
        error!("Fatal error occurred: {}", e);
        Err(e)
    } else {
        info!("Success");
        Ok(())
    }
}

fn run(matches: &ArgMatches) -> Result<()> {
    if matches.get_flag("profiles-list") {
        config::print_profiles()?;
    } else {
        let config = Config::try_from(matches)?;

        // Create zip file
        match config.container {
            ContainerType::None => save(container::Plain::new(&config)?, &config)?,
            ContainerType::Zip => save(container::Zip::new(&config)?, &config)?,
            ContainerType::SevenZip => save(container::SevenZip::new(&config)?, &config)?,
            ContainerType::Tar => match config.compression {
                CompressionType::None => save(container::Tar::new(&config)?, &config)?,
                CompressionType::Deflate => bail!("No deflate compression in zip"),
                CompressionType::Bzip2 => save(container::Tar::new_bz2(&config)?, &config)?,
                CompressionType::Bzip3 => save(container::Tar::new_bz3(&config)?, &config)?,
                CompressionType::Zstd => save(container::Tar::new_zstd(&config)?, &config)?,
                CompressionType::Gzip => save(container::Tar::new_gzip(&config)?, &config)?,
                CompressionType::Xz => save(container::Tar::new_xz(&config)?, &config)?,
            },
        };
    }
    Ok(())
}

fn save(mut container: impl container::Container, config: &Config) -> Result<()> {
    debug!(
        "Run with config {:?} on {} to {}",
        config,
        &config.save_dir,
        config.output.display()
    );

    let save_current_dir = current_dir()?;

    let multi_progress_bar = indicatif::MultiProgress::new();
    let record_progress_bar = multi_progress_bar.add(
        ProgressBar::new(config.records.len() as u64).with_style(
            ProgressStyle::with_template("  [{elapsed:6}] Record {pos} / {len} {wide_bar}")
                .unwrap(),
        ),
    );

    info!("Save the code");
    for record in config.records.iter().progress_with(record_progress_bar) {
        let output_can = config.output.canonicalize()?;

        // Set current dir
        if let Some(dir) = &record.directory {
            set_current_dir(dir.to_logical_path(&save_current_dir))?;
        }

        // Create Walker
        if !record.paths.is_empty() {
            // Create override
            let mut walker_override_builder =
                ignore::overrides::OverrideBuilder::new(current_dir()?);
            for whitelist in &record.whitelist {
                walker_override_builder.add(whitelist)?;
            }
            for ignore in &record.ignore {
                walker_override_builder.add(&format!("!{}", ignore))?;
            }

            let mut walker = WalkBuilder::new(&record.paths[0].to_logical_path("."));
            walker
                .same_file_system(true)
                .skip_stdout(true)
                .follow_links(false)
                .hidden(!record.hidden)
                .git_ignore(record.gitignore)
                .overrides(walker_override_builder.build()?)
                // Filter output file
                .filter_entry(move |entry| {
                    if entry.path().file_name().unwrap() == output_can.file_name().unwrap() {
                        if let Ok(can_entry_path) = entry.path().canonicalize() {
                            can_entry_path != output_can
                        } else {
                            true
                        }
                    } else {
                        true
                    }
                });
            for path in record.paths.iter().skip(1) {
                walker.add(path.to_logical_path("."));
            }

            // Progress bar
            let progress_bar = if config.verbosity >= 1 {
                ProgressBar::hidden()
            } else {
                ProgressBar::new_spinner()
            };
            progress_bar.set_style(
                ProgressStyle::with_template(
                    "{spinner} [{elapsed:6}] Current file {human_pos:10} : Speed {per_sec}",
                )
                .unwrap(),
            );
            let walk_progress_bar = multi_progress_bar.add(progress_bar);

            // Save dirs
            debug!("Walk dir");
            for result in walker.build().progress_with(walk_progress_bar) {
                match result {
                    Ok(entry) => {
                        debug!("Save entry '{}'", entry.path().display());
                        if let Err(e) =
                            container.add_path(RelativePath::new(&entry.path().to_slash_lossy()))
                        {
                            warn!("Cannot add '{}': {}", entry.path().display(), e);
                        }
                    }
                    Err(err) => warn!("Cannot read path: {}", err),
                }
            }
        }

        // Add files
        for filename in &record.files {
            debug!("Save file '{}'", filename);
            if let Err(e) = container.add_path(filename) {
                warn!("Cannot add '{}' in zip: {}", filename, e);
            }
        }

        // Reset current dir
        if record.directory.is_some() {
            set_current_dir(&save_current_dir)?;
        }
    }

    container.finish()?;

    // Add signatures
    for signature in &config.signatures {
        signature.generate_signature(&config.output)?;
    }

    Ok(())
}
