use anyhow::{anyhow, bail, Context, Result};
use chrono::Local;
use clap::{value_parser, Arg, ArgAction, ArgMatches, Command};
use itertools::Itertools;
use log::{info, warn};
use path_slash::PathExt;
use relative_path::RelativePathBuf;
use serde::Deserialize;
use std::collections::HashMap;
use std::env::{current_dir, set_current_dir};
use std::fmt::Display;
use std::fs;
use std::io::{BufReader, Write};
use std::path::Path;
use std::str::FromStr;
use std::{fs::File, path::PathBuf};
use strum::VariantNames;
use strum_macros::{Display, EnumString, EnumVariantNames};

pub fn create_app() -> Command {
    Command::new(env!("CARGO_PKG_NAME"))
        .version(env!("CARGO_PKG_VERSION"))
        .author(env!("CARGO_PKG_AUTHORS"))
        .about(env!("CARGO_PKG_DESCRIPTION"))
        .arg(
            Arg::new("verbosity")
                .short('v')
                .long("verbose")
                .action(ArgAction::Count)
                .help("Sets the level of verbosity"),
        )
        .arg(
            Arg::new("output")
                .short('o')
                .long("output")
                .value_name("OUT_NAME")
                .help("Output directory or filename")
                .value_parser(value_parser!(PathBuf))
        )
        .arg(
            Arg::new("profiles-list")
                .short('r')
                .long("profiles-list")
                .action(ArgAction::SetTrue)
                .help("List profiles")
        )
        .arg(
            Arg::new("paths")
                .short('p')
                .long("paths")
                .value_name("PATH")
                .help("Paths to save")
                .action(ArgAction::Append)
        )
        .arg(
            Arg::new("files")
                .short('f')
                .long("files")
                .value_name("FILE")
                .help("Files to save")
                .action(ArgAction::Append)
        )
        .arg(
            Arg::new("container")
                .short('t')
                .long("container")
                .value_name("CONTAINER")
                .help("Container type")
                .value_parser(ContainerType::VARIANTS.to_vec())
        )
        .arg(
            Arg::new("compression")
                .short('c')
                .long("compression")
                .value_name("COMPRESSION")
                .value_parser(CompressionType::VARIANTS.to_vec())
                .help("Compression method")
        )
        .arg(
            Arg::new("compression_level")
                .short('l')
                .long("comp-level")
                .value_name("COMP_LEVEL")
                .help("Compression level")
                .long_help("Deflated: 0 - 9 (default is 6)\nBzip2: 0 - 9 (default is 6)\nZstd: 1 - 21 (default is 3)\nGzip: 0 - 9 (default is 6)\nXz: 0 - 9 (default is 6)")
                .value_parser(|val: &str| i32::from_str(&val).map(|_| ()).map_err(|e| e.to_string() ) )
        )
        .arg(
            Arg::new("hidden")
                .short('e')
                .long("hidden")
                .action(ArgAction::SetTrue)
                .help("Save also hidden file (begin with .)"),
        )
        .arg(
            Arg::new("gitignore")
                .short('g')
                .long("no-gitignore")
                .action(ArgAction::SetTrue)
                .help("Do not follow gitignore rules"),
        )
        .arg(
            Arg::new("ignore")
                .short('i')
                .long("ignore")
                .value_name("IGNORE")
                .help("gitignore like line")
                .action(clap::ArgAction::Append)
        )
        .arg(
            Arg::new("whitelist")
                .short('w')
                .long("whitelist")
                .value_name("WHITELIST")
                .help("gitignore like whitelist line")
                .action(clap::ArgAction::Append)
        )
        .arg(
            Arg::new("signatures")
                .short('s')
                .long("signature")
                .value_name("SIGN")
                .help("Signatures to generate")
                .value_parser(Signatures::VARIANTS.to_vec())
                .action(ArgAction::Append)
        )
        .arg(
            Arg::new("directory")
                .short('d')
                .long("directory")
                .value_name("DIR")
                .help("Current directory")
                .value_parser(value_parser!(PathBuf))
        )
        .arg(
            Arg::new("profiles")
                .value_name("CONFIG")
                .help("Profiles used found in smc.toml")
                .action(ArgAction::Append),
        )
}

#[derive(Copy, Clone, Debug, EnumString, Deserialize, EnumVariantNames, Display)]
#[strum(ascii_case_insensitive)]
pub enum ContainerType {
    None,
    Zip,
    SevenZip,
    Tar,
}

impl ContainerType {
    const fn extension(self, compression: CompressionType) -> &'static str {
        match self {
            ContainerType::None => "",
            ContainerType::Zip => ".zip",
            ContainerType::SevenZip => ".7z",
            ContainerType::Tar => match compression {
                CompressionType::None => ".tar",
                CompressionType::Deflate => ".tar",
                CompressionType::Bzip2 => ".tar.bz2",
                CompressionType::Bzip3 => ".tar.bz3",
                CompressionType::Zstd => ".tar.zst",
                CompressionType::Gzip => ".tar.gz",
                CompressionType::Xz => ".tar.xz",
            },
        }
    }
}

#[derive(Copy, Clone, Debug, EnumString, Deserialize, EnumVariantNames, Display)]
#[strum(ascii_case_insensitive)]
pub enum CompressionType {
    None,
    Deflate,
    Bzip2,
    Bzip3,
    Zstd,
    Gzip,
    Xz,
}

impl TryFrom<CompressionType> for zip::CompressionMethod {
    type Error = anyhow::Error;

    fn try_from(value: CompressionType) -> Result<Self> {
        match value {
            CompressionType::None => Ok(zip::CompressionMethod::Stored),
            CompressionType::Deflate => Ok(zip::CompressionMethod::Deflated),
            CompressionType::Bzip2 => Ok(zip::CompressionMethod::Bzip2),
            CompressionType::Bzip3 => bail!("No bzip3 compression in zip"),
            CompressionType::Zstd => Ok(zip::CompressionMethod::Zstd),
            CompressionType::Gzip => bail!("No gzip compression in zip"),
            CompressionType::Xz => bail!("No xz compression in zip"),
        }
    }
}

#[derive(Copy, Clone, Debug, EnumString, Deserialize, EnumVariantNames, Display)]
#[strum(ascii_case_insensitive)]
pub enum Signatures {
    Sha256,
    Sha384,
    Sha512,
    Sha3_256,
    Sha3_384,
    Sha3_512,
}

impl Signatures {
    pub fn generate_signature(&self, path: &Path) -> Result<()> {
        match self {
            Signatures::Sha256 => generate_hash::<sha2::Sha256>(path, ".sha256sum"),
            Signatures::Sha384 => generate_hash::<sha2::Sha384>(path, ".sha384sum"),
            Signatures::Sha512 => generate_hash::<sha2::Sha512>(path, ".sha512sum"),
            Signatures::Sha3_256 => generate_hash::<sha3::Sha3_256>(path, ".sha3_256sum"),
            Signatures::Sha3_384 => generate_hash::<sha3::Sha3_384>(path, ".sha3_384sum"),
            Signatures::Sha3_512 => generate_hash::<sha3::Sha3_512>(path, ".sha3_512sum"),
        }
    }
}

fn generate_hash<D: digest::Digest + Write>(path: &Path, extension: &str) -> Result<()> {
    let mut file = BufReader::new(File::open(path)?);
    let mut hasher = D::new();
    std::io::copy(&mut file, &mut hasher)?;
    let output = hasher.finalize();
    let mut output_filename = path.to_path_buf();
    output_filename.set_file_name(format!(
        "{}{}",
        path.file_name().unwrap().to_string_lossy(),
        extension
    ));
    let mut out_file = File::create(output_filename)?;
    write!(
        out_file,
        "{} {}",
        hex::encode(output),
        path.file_name().unwrap().to_string_lossy()
    )?;
    Ok(())
}

#[derive(Debug, Deserialize)]
struct Profile {
    #[serde(default)]
    paths: Vec<RelativePathBuf>,
    output: Option<PathBuf>,
    compression_level: Option<i32>,
    hidden: Option<bool>,
    gitignore: Option<bool>,
    container: Option<ContainerType>,
    compression: Option<CompressionType>,
    directory: Option<RelativePathBuf>,
    #[serde(default)]
    files: Vec<RelativePathBuf>,
    #[serde(default)]
    signatures: Vec<Signatures>,
    #[serde(default)]
    ignore: Vec<String>,
    #[serde(default)]
    whitelist: Vec<String>,
}

impl Default for Profile {
    fn default() -> Self {
        Profile {
            paths: vec![],
            output: Some(PathBuf::from(".")),
            compression_level: None,
            hidden: Some(false),
            gitignore: Some(true),
            container: Some(ContainerType::Zip),
            compression: Some(CompressionType::Deflate),
            directory: None,
            files: vec![],
            signatures: vec![],
            ignore: vec![],
            whitelist: vec![],
        }
    }
}

impl Profile {
    fn or_default(mut self) -> Self {
        let default = Self::default();
        if self.output.is_none() {
            self.output = default.output;
        }
        if self.hidden.is_none() {
            self.hidden = default.hidden;
        }
        if self.gitignore.is_none() {
            self.gitignore = default.gitignore;
        }
        if self.container.is_none() {
            self.container = default.container;
        }
        if self.compression.is_none() {
            self.compression = default.compression;
        }
        self
    }

    fn add_cmd(mut self, matches: &ArgMatches) -> Self {
        // Add save filters
        if matches.get_flag("hidden") {
            self.hidden = Some(true);
        }
        if matches.get_flag("gitignore") {
            self.gitignore = Some(false);
        }

        // Add ignore
        if let Some(ignores) = matches.get_many::<String>("ignore") {
            self.ignore.extend(ignores.map(|i| i.clone()));
        }

        // Add whitelist
        if let Some(whitelists) = matches.get_many::<String>("whitelist") {
            self.whitelist.extend(whitelists.map(|w| w.clone()));
        }

        self
    }
}

macro_rules! print_attr {
    ($profile_name:ident, disp: $attr_name:ident ($printed_name:expr), $formatter_name:ident, $first_char:ident) => {
        if let Some($attr_name) = &$profile_name.$attr_name {
            writeln!(
                $formatter_name,
                "{}   ├ {}: {}",
                $first_char, $printed_name, $attr_name
            )?;
        }
    };
    ($profile_name:ident, path: $attr_name:ident ($printed_name:expr), $formatter_name:ident, $first_char:ident) => {
        if let Some($attr_name) = &$profile_name.$attr_name {
            writeln!(
                $formatter_name,
                "{}   ├ {}: {}",
                $first_char,
                $printed_name,
                $attr_name.display()
            )?;
        }
    };
    ($profile_name:ident, disp_list: $attr_name:ident ($printed_name:expr), $formatter_name:ident, $first_char:ident) => {
        if !$profile_name.$attr_name.is_empty() {
            writeln!($formatter_name, "{}   ├ {}", $first_char, $printed_name)?;
            for (pos, val) in $profile_name.$attr_name.iter().with_position() {
                match pos {
                    itertools::Position::First | itertools::Position::Middle => {
                        writeln!($formatter_name, "{}   │ ├ {}", $first_char, val)?
                    }
                    itertools::Position::Last | itertools::Position::Only => {
                        writeln!($formatter_name, "{}   │ └ {}", $first_char, val)?
                    }
                }
            }
        }
    };
}

impl Display for Profile {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        let first_char = if f.sign_plus() { '│' } else { ' ' };

        print_attr!(self, path: output("output"), f, first_char);
        print_attr!(self, disp: compression_level("compression level"), f, first_char);
        print_attr!(self, disp: hidden("hidden"), f, first_char);
        print_attr!(self, disp: gitignore("gitignore"), f, first_char);
        print_attr!(self, disp: container("container"), f, first_char);
        print_attr!(self, disp: compression("compression"), f, first_char);
        print_attr!(self, disp: directory("directory"), f, first_char);
        print_attr!(self, disp_list: signatures("signatures"), f, first_char);
        print_attr!(self, disp_list: ignore("ignore"), f, first_char);
        print_attr!(self, disp_list: whitelist("whitelist"), f, first_char);

        writeln!(f, "{}   └ paths", first_char,)?;
        for (pos, path) in self.paths.iter().chain(self.files.iter()).with_position() {
            match pos {
                itertools::Position::First | itertools::Position::Middle => {
                    writeln!(f, "{}     ├ path: {}", first_char, path)?
                }
                itertools::Position::Last | itertools::Position::Only => {
                    write!(f, "{}     └ path: {}", first_char, path)?
                }
            }
        }
        Ok(())
    }
}

#[derive(Debug, Deserialize)]
struct ProfileList {
    profiles: HashMap<String, Profile>,
}

impl ProfileList {
    fn get_similar_profile(&self, original_profile_name: &str) -> Option<&str> {
        // Get the more similar name base on jaro winkler
        let max = self
            .profiles
            .keys()
            .map(|profile_name| {
                let sim = strsim::jaro_winkler(original_profile_name, profile_name);
                (sim, profile_name)
            })
            .max_by(|(sim_1, _), (sim_2, _)| {
                sim_1.partial_cmp(sim_2).unwrap_or(std::cmp::Ordering::Less)
            });

        // If something similar found, use it
        if let Some((sim, profile_name)) = max {
            if sim > 0.5 {
                Some(profile_name)
            } else {
                None
            }
        } else {
            None
        }
    }
}

pub fn print_profiles() -> Result<()> {
    let profile_list: ProfileList =
        toml::from_str(&fs::read_to_string("smc.toml").with_context(|| "No config file")?)
            .context("Invalid config file")?;
    println!("{}", profile_list);
    Ok(())
}

impl Display for ProfileList {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        writeln!(f, "profiles")?;
        for (profile_pos, (name, profile)) in self.profiles.iter().with_position() {
            match profile_pos {
                itertools::Position::Middle => {
                    writeln!(f, "├── {}", name)?;
                    writeln!(f, "{:+}", profile)?;
                }
                itertools::Position::First => {
                    writeln!(f, "├── {}", name)?;
                    writeln!(f, "{:+}", profile)?;
                }
                itertools::Position::Last => {
                    writeln!(f, "└── {}", name)?;
                    write!(f, "{}", profile)?;
                }
                itertools::Position::Only => {
                    writeln!(f, "└── {}", name)?;
                    write!(f, "{}", profile)?;
                }
            }
        }
        Ok(())
    }
}

#[derive(Debug)]
pub struct RecordConfig {
    pub paths: Vec<RelativePathBuf>,
    pub hidden: bool,
    pub gitignore: bool,
    pub ignore: Vec<String>,
    pub whitelist: Vec<String>,
    pub directory: Option<RelativePathBuf>,
    pub files: Vec<RelativePathBuf>,
}

impl From<Profile> for RecordConfig {
    fn from(profile: Profile) -> Self {
        RecordConfig {
            paths: profile.paths,
            hidden: profile.hidden.unwrap(),
            gitignore: profile.gitignore.unwrap(),
            directory: profile.directory,
            files: profile.files,
            ignore: profile.ignore,
            whitelist: profile.whitelist,
        }
    }
}

#[derive(Debug)]
pub struct Config {
    pub container: ContainerType,
    pub output: PathBuf,
    pub save_dir: String,
    pub verbosity: i32,
    pub compression: CompressionType,
    pub compression_level: Option<i32>,
    pub signatures: Vec<Signatures>,
    pub records: Vec<RecordConfig>,
}

impl Config {
    fn add_cmd(&mut self, matches: &ArgMatches) -> Result<()> {
        // Verbosity
        self.verbosity = matches.get_count("verbosity") as i32;

        // Container
        if let Some(container) = matches.get_one("container") {
            self.container = *container;
        }

        // Add compression
        if let Some(compression_level) = matches.get_one("compression_level") {
            self.compression_level = Some(*compression_level);
        }
        if let Some(compression) = matches.get_one("compression") {
            self.compression = *compression
        }

        // Add output
        if let Some(output_dir) = matches.get_one::<PathBuf>("output") {
            self.output = output_dir.clone()
        }

        // Add paths
        if let Some(paths) = matches.get_raw("paths") {
            self.records[0].paths.extend(
                paths.map(|f| RelativePathBuf::from(Path::new(f).to_slash_lossy().as_ref())),
            );
        }

        // Add files
        if let Some(files) = matches.get_raw("files") {
            self.records[0].files.extend(
                files.map(|f| RelativePathBuf::from(Path::new(f).to_slash_lossy().as_ref())),
            );
        }

        // Add signatures
        if let Some(files) = matches.get_many::<Signatures>("signatures") {
            self.signatures.extend(files.map(|f| *f));
        }

        // Current dir
        self.save_dir = current_dir()
            .unwrap()
            .file_name()
            .unwrap()
            .to_string_lossy()
            .to_string();

        // Create output filename
        if self.output.is_dir() {
            self.output = self.output.join("${SAVE_DIR}_${TIME}${EXT}");
        }

        // Expand
        self.output = shellexpand::path::env_with_context_no_errors(&self.output, |s| match s {
            "TIME" => Some(Local::now().format("%Y-%m-%d_%H-%M-%S").to_string()),
            "SAVE_DIR" => Some(self.save_dir.clone()),
            "EXT" => Some(self.container.extension(self.compression).to_string()),
            _ => {
                warn!("Unknown variable {}", s);
                None
            }
        })
        .to_path_buf();

        Ok(())
    }
}

impl FromIterator<Profile> for Result<Config> {
    fn from_iter<T: IntoIterator<Item = Profile>>(iter: T) -> Self {
        let mut profile_iter = iter.into_iter();
        let first_profile = profile_iter.next().ok_or_else(|| anyhow!("No profile"))?;

        let container = first_profile.container.unwrap();
        let compression = first_profile.compression.unwrap();
        let compression_level = first_profile.compression_level;
        let output = first_profile.output.as_ref().unwrap().clone();
        let signatures = first_profile.signatures.clone();
        let records = vec![RecordConfig::from(first_profile)];

        let mut config = Config {
            container,
            output,
            save_dir: String::new(),
            verbosity: 0,
            compression,
            compression_level,
            signatures,
            records,
        };

        config
            .records
            .extend(profile_iter.map(|profile| profile.into()));

        Ok(config)
    }
}

impl<'a> TryFrom<&ArgMatches> for Config {
    type Error = anyhow::Error;

    fn try_from(matches: &ArgMatches) -> Result<Self> {
        // Set current dir
        if let Some(directory) = matches.get_one::<PathBuf>("directory") {
            set_current_dir(directory)?;
        }

        let mut config = if matches.contains_id("profiles") {
            let mut profile_list: ProfileList =
                toml::from_str(&fs::read_to_string("smc.toml").with_context(|| "No config file")?)
                    .context("Invalid config file")?;

            Result::<Config>::from_iter(
                matches
                    .get_many::<String>("profiles")
                    .unwrap()
                    .filter_map(|profile_name| {
                        let profile = profile_list.profiles.remove(profile_name);
                        if profile.is_none() {
                            warn!("Profile `{}` not found", profile_name);
                            if let Some(profile_similar) =
                                profile_list.get_similar_profile(profile_name)
                            {
                                info!("Did you mean `{}` ?", profile_similar);
                            }
                        }
                        profile.map(|config| config.add_cmd(matches).or_default())
                    }),
            )?
        } else {
            Result::<Config>::from_iter(std::iter::once(Profile::default().add_cmd(matches)))
                .unwrap()
        };

        config.add_cmd(matches)?;

        Ok(config)
    }
}
